<?php
// pages/print_supplier_report.php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/functions.php';

// Only allow users with 'view_suppliers' permission or super admin
if (!hasPermission('view_suppliers') && !$isSuperAdmin) {
    header('Location: ../pages/dashboard.php');
    exit;
}

// Get supplier ID from GET
$supplier_id = isset($_GET['supplier_id']) ? intval($_GET['supplier_id']) : 0;
if (!$supplier_id) {
    die('Invalid supplier ID');
}

// Fetch store info
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group = 'store_info'");
$stmt->execute();
$storeSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
$store = [
    'logo' => $storeSettings['store_logo'] ?? '',
    'name' => $storeSettings['store_name'] ?? 'Store',
    'address' => $storeSettings['store_address'] ?? '',
    'phone' => $storeSettings['store_phone'] ?? '',
    'email' => $storeSettings['store_email'] ?? '',
    'website' => $storeSettings['website'] ?? '',
];

// Fetch supplier info
$stmt = $pdo->prepare("SELECT * FROM suppliers WHERE id = ?");
$stmt->execute([$supplier_id]);
$supplier = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$supplier) {
    die('Supplier not found');
}

// Fetch all receipts/items for this supplier, including category
$stmt = $pdo->prepare("
    SELECT sr.*, p.name as product_name, c.name as category_name
    FROM supplier_receipts sr
    JOIN products p ON sr.product_id = p.id
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE sr.supplier_id = ?
    ORDER BY c.name, sr.created_at DESC
");
$stmt->execute([$supplier_id]);
$items = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate totals
$total_amount = 0;
$total_paid = 0;
foreach ($items as $item) {
    $line_total = $item['quantity'] * $item['unit_price'];
    $total_amount += $line_total;
    if ($item['payment_status'] === 'paid') {
        $total_paid += $line_total;
    }
}
$total_balance = $total_amount - $total_paid;

// Group items by category
$items_by_category = [];
foreach ($items as $item) {
    $cat = $item['category_name'] ?: 'Uncategorized';
    if (!isset($items_by_category[$cat])) {
        $items_by_category[$cat] = [];
    }
    $items_by_category[$cat][] = $item;
}

?><!DOCTYPE html>
<html>
<head>
    <title>Supplier Report - <?php echo htmlspecialchars($supplier['name']); ?></title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Inter', Arial, sans-serif; background: #f4f7fa; color: #222; margin: 0; padding: 0; }
        .main-card { background: #fff; border-radius: 18px; box-shadow: 0 4px 24px rgba(37,99,235,0.07); max-width: 900px; margin: 32px auto 24px auto; padding: 32px 36px 28px 36px; }
        .header { text-align: center; margin-bottom: 24px; }
        .logo { max-height: 70px; margin-bottom: 10px; border-radius: 8px; box-shadow: 0 2px 8px rgba(37,99,235,0.08); }
        .store-title { font-size: 2em; color: #2563eb; font-weight: 700; margin: 0 0 6px 0; letter-spacing: 1px; }
        .store-contact { font-size: 1.05em; color: #444; margin-bottom: 2px; }
        .store-website { color: #2563eb; text-decoration: underline; }
        .info-table { width: 100%; margin-bottom: 18px; border-radius: 8px; background: #f1f5f9; }
        .info-table td { padding: 7px 12px; font-size: 1.08em; }
        .section-title { font-weight: 700; margin: 28px 0 12px 0; font-size: 1.18em; color: #2563eb; letter-spacing: 0.5px; }
        .summary-cards { display: flex; gap: 18px; margin-bottom: 24px; justify-content: center; }
        .summary-card {
            flex: 1;
            min-width: 150px;
            background: linear-gradient(120deg, #2563eb 0%, #60a5fa 100%);
            border-radius: 14px;
            padding: 18px 0 12px 0;
            text-align: center;
            box-shadow: 0 2px 12px rgba(37,99,235,0.09);
            color: #fff;
            position: relative;
            transition: transform 0.15s, box-shadow 0.15s;
        }
        .summary-card:hover {
            transform: translateY(-2px) scale(1.03);
            box-shadow: 0 6px 24px rgba(37,99,235,0.13);
        }
        .summary-icon {
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 2em;
            margin-bottom: 6px;
            color: #fff;
            opacity: 0.92;
        }
        .summary-label {
            font-size: 1.04em;
            color: #e0e7ff;
            font-weight: 600;
            margin-bottom: 4px;
            letter-spacing: 0.5px;
        }
        .summary-value {
            font-size: 1.45em;
            color: #fff;
            font-weight: 700;
            letter-spacing: 0.5px;
        }
        .items-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0;
            margin-bottom: 18px;
            font-size: 0.93em;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(37,99,235,0.04);
            table-layout: fixed;
        }
        .items-table th, .items-table td {
            padding: 4px 6px;
            text-align: left;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        .items-table th {
            background: #2563eb;
            color: #fff !important;
            font-weight: 700;
            border: none;
            font-size: 0.97em;
        }
        .items-table td.serial-list {
            max-width: 120px;
            word-break: break-all;
            white-space: normal;
            font-size: 0.92em;
        }
        .items-table td {
            font-size: 0.92em;
        }
        .items-table tr.category-header td {
            background: #e0e7ff;
            color: #1e40af;
            font-weight: 700;
            font-size: 1.01em;
            border-top: 2px solid #2563eb;
        }
        .items-table tbody tr:nth-child(even):not(.category-header) { background: #f8fafc; }
        .items-table tbody tr:hover:not(.category-header) { background: #e0e7ff33; }
        .serial-list { font-size: 0.97em; color: #444; }
        .badge-status { display: inline-block; padding: 2px 8px; border-radius: 12px; font-size: 0.92em; font-weight: 600; }
        .badge-paid { background: #22c55e; color: #fff; }
        .badge-credit { background: #f59e42; color: #fff; }
        .badge-other { background: #64748b; color: #fff; }
        .footer { text-align: center; margin-top: 32px; font-size: 1.05em; color: #888; }
        .fab-print { position: fixed; bottom: 32px; right: 32px; background: #2563eb; color: #fff; border: none; border-radius: 50%; width: 56px; height: 56px; box-shadow: 0 4px 16px rgba(37,99,235,0.18); font-size: 2em; display: flex; align-items: center; justify-content: center; z-index: 1000; cursor: pointer; transition: background 0.2s; }
        .fab-print:hover { background: #1e40af; }
        .main-card { overflow-x: auto; }
        @media (max-width: 1100px) {
            .items-table { font-size: 0.85em; }
            .items-table th, .items-table td { padding: 3px 4px; }
        }
        @media (max-width: 900px) {
            .main-card { padding: 12px 2px; }
            .items-table { font-size: 0.8em; }
        }
        @media print {
            body { background: #fff; }
            .main-card { box-shadow: none; margin: 0; padding: 0; border-radius: 0; }
            .fab-print, .no-print { display: none !important; }
            .header, .footer { color: #000; }
            .main-card, .items-table th, .items-table td, .summary-card, .supplier-info-card {
                -webkit-print-color-adjust: exact !important;
                print-color-adjust: exact !important;
                color-adjust: exact !important;
            }
            .items-table th {
                color: #fff !important;
            }
            .items-table th, .items-table td {
                color: #222 !important;
            }
            .summary-card, .supplier-info-card {
                color: #222 !important;
            }
        }
        .supplier-info-card {
            background: linear-gradient(90deg, #e0e7ff 0%, #f1f5f9 100%);
            border-radius: 14px;
            padding: 10px 28px 6px 28px;
            margin-bottom: 28px;
            box-shadow: 0 2px 12px rgba(37,99,235,0.07);
            max-width: 700px;
            margin-left: auto;
            margin-right: auto;
            border: 1.5px solid #dbeafe;
        }
        .supplier-info-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 8px 24px;
        }
        .supplier-info-row {
            display: flex;
            align-items: center;
            margin-bottom: 0;
            background: none;
            border-radius: 8px;
            padding: 2px 0;
        }
        .supplier-info-label {
            font-weight: 600;
            color: #2563eb;
            display: flex;
            align-items: center;
            font-size: 1em;
            min-width: 120px;
        }
        .supplier-info-value {
            color: #222;
            font-size: 1em;
            font-weight: 500;
            padding-left: 8px;
            word-break: break-word;
        }
        .info-icon {
            font-size: 1.1em;
            margin-right: 7px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            width: 28px;
            height: 28px;
            border-radius: 50%;
            background: #dbeafe;
            color: #2563eb;
        }
        @media (max-width: 700px) {
            .supplier-info-card { padding: 8px 4px; }
            .supplier-info-grid { grid-template-columns: 1fr; }
        }
        .items-table td.date-col, .items-table th.date-col {
            font-size: 0.85em;
            min-width: 110px;
            max-width: 130px;
            color: #2563eb;
            font-weight: 600;
            word-break: break-word;
            white-space: normal;
        }
        .footer-date {
            font-size: 0.92em;
            color: #2563eb;
        }
        .items-table th.date-col {
            color: #fff !important;
        }
        .items-table td.product-col, .items-table th.product-col,
        .items-table td.serial-col, .items-table th.serial-col {
            white-space: normal;
            word-break: break-word;
            max-width: 180px;
        }
    </style>
</head>
<body>
    <div class="main-card">
    <div class="header">
        <?php if ($store['logo']): ?>
            <img src="<?php echo htmlspecialchars($store['logo']); ?>" class="logo"><br>
        <?php endif; ?>
            <div class="store-title"><?php echo htmlspecialchars($store['name']); ?></div>
            <div class="store-contact"><?php echo htmlspecialchars($store['address']); ?></div>
            <div class="store-contact">Phone: <?php echo htmlspecialchars($store['phone']); ?> | Email: <?php echo htmlspecialchars($store['email']); ?></div>
        <?php if (!empty($store['website'])): ?>
                <div class="store-contact">Website: <a href="<?php echo htmlspecialchars($store['website']); ?>" class="store-website" target="_blank"><?php echo htmlspecialchars($store['website']); ?></a></div>
        <?php endif; ?>
    </div>
        <div class="supplier-info-card">
            <div class="supplier-info-grid">
                <div class="supplier-info-row">
                    <div class="supplier-info-label"><span class="info-icon">🏢</span>Supplier:</div>
                    <div class="supplier-info-value"><?php echo htmlspecialchars($supplier['name']); ?></div>
                </div>
                <div class="supplier-info-row">
                    <div class="supplier-info-label"><span class="info-icon">👤</span>Contact Person:</div>
                    <div class="supplier-info-value"><?php echo htmlspecialchars($supplier['contact_person']); ?></div>
                </div>
                <div class="supplier-info-row">
                    <div class="supplier-info-label"><span class="info-icon">📞</span>Phone:</div>
                    <div class="supplier-info-value"><?php echo htmlspecialchars($supplier['phone']); ?></div>
                </div>
                <div class="supplier-info-row">
                    <div class="supplier-info-label"><span class="info-icon">✉️</span>Email:</div>
                    <div class="supplier-info-value"><?php echo htmlspecialchars($supplier['email']); ?></div>
                </div>
                <div class="supplier-info-row" style="grid-column: 1 / -1;">
                    <div class="supplier-info-label"><span class="info-icon">📍</span>Address:</div>
                    <div class="supplier-info-value"><?php echo htmlspecialchars($supplier['address']); ?></div>
                </div>
            </div>
        </div>
    <div class="section-title">Summary</div>
        <div class="summary-cards">
            <div class="summary-card">
                <div class="summary-icon">💰</div>
                <div class="summary-label">Total Amount</div>
                <div class="summary-value"><?php echo number_format($total_amount, 2); ?></div>
            </div>
            <div class="summary-card">
                <div class="summary-icon">✅</div>
                <div class="summary-label">Total Paid</div>
                <div class="summary-value"><?php echo number_format($total_paid, 2); ?></div>
            </div>
            <div class="summary-card">
                <div class="summary-icon">🧾</div>
                <div class="summary-label">Total Balance</div>
                <div class="summary-value"><?php echo number_format($total_balance, 2); ?></div>
            </div>
        </div>
    <div class="section-title">Items & Payments</div>
    <table class="items-table">
        <thead>
            <tr>
                    <th class="date-col">Date</th>
                <th class="product-col">Product</th>
                <th class="serial-col">Serial No:</th>
                <th>Qty</th>
                <th>Unit Price</th>
                <th>Total</th>
                    <th>Paid</th>
                    <th>Balance</th>
                    <th>Status</th>
                    <th class="date-col">Date</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($items_by_category as $cat => $cat_items): ?>
                    <tr class="category-header"><td colspan="10"><?php echo htmlspecialchars($cat); ?></td></tr>
                    <?php foreach ($cat_items as $item):
                        $line_total = $item['quantity'] * $item['unit_price'];
                        $amount_paid = ($item['payment_status'] === 'paid') ? $line_total : 0;
                        $balance = $line_total - $amount_paid;
                        // Fetch all serial numbers for this item from product_serial_numbers
                        $serials = [];
                        if (!empty($item['id']) && !empty($item['product_id'])) {
                            $stmtSerials = $pdo->prepare("
                                SELECT GROUP_CONCAT(serial_number ORDER BY created_at ASC) as serial_numbers
                                FROM product_serial_numbers 
                                WHERE product_id = ? 
                                AND source_id = ?
                            ");
                            $stmtSerials->execute([$item['product_id'], $item['id']]);
                            $serialResult = $stmtSerials->fetch(PDO::FETCH_ASSOC);
                            if ($serialResult && $serialResult['serial_numbers']) {
                                $serials = explode(',', $serialResult['serial_numbers']);
                            }
                        }
                    ?>
                    <tr>
                        <td class="date-col"><?php echo date('j/n/Y H:i', strtotime($item['created_at'])); ?></td>
                    <td class="product-col"><?php echo htmlspecialchars($item['product_name']); ?></td>
                    <td class="serial-col">
                        <?php
                            if (!empty($serials)) {
                                echo '<span class="serial-list">' . implode(', ', array_map('htmlspecialchars', $serials)) . '</span>';
                        } else {
                            echo '-';
                        }
                        ?>
                    </td>
                    <td><?php echo number_format($item['quantity'], 2); ?></td>
                    <td><?php echo number_format($item['unit_price'], 2); ?></td>
                        <td><?php echo number_format($line_total, 2); ?></td>
                        <td><?php echo number_format($amount_paid, 2); ?></td>
                        <td><?php echo number_format($balance, 2); ?></td>
                        <td>
                            <?php
                            $status = strtolower($item['payment_status']);
                            if ($status === 'paid') {
                                echo '<span class="badge-status badge-paid">Paid</span>';
                            } elseif ($status === 'credit') {
                                echo '<span class="badge-status badge-credit">Credit</span>';
                            } else {
                                echo '<span class="badge-status badge-other">' . htmlspecialchars(ucfirst($status)) . '</span>';
                            }
                            ?>
                        </td>
                        <td class="date-col">
                        <?php
                        if (!empty($item['updated_at']) && $item['payment_status'] === 'paid') {
                                echo date('j/n/Y H:i', strtotime($item['updated_at']));
                        } else {
                            echo '-';
                        }
                        ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            <?php endforeach; ?>
        </tbody>
    </table>
    <div class="footer">
            <span class="footer-date">Printed on <?php echo date('j/n/Y H:i'); ?></span>
        </div>
    </div>
    <button class="fab-print no-print" onclick="triggerPrint()" title="Print"><span style="font-size:1.5em;">🖨️</span></button>
    <script>
        function triggerPrint() {
            document.body.classList.remove('a4-print');
            document.body.classList.add('a4-print');
            setTimeout(function() { window.print(); }, 100);
        }
        // Automatically open print dialog when page loads
        document.addEventListener('DOMContentLoaded', function() {
            triggerPrint();
        });
    </script>
</body>
</html> 